<?php
//***********************************************************************/
// chooser.class.php                                                    */
// ===========================                                          */
// Copyright (c) 2004 by David Rolston  (gizmo@gizmola.com)             */
// http://www.gizmola.com                                               */
//                                                                      */
// This program is free software. You can redistribute it and/or modify */
// it under the terms of the GNU General Public License as published by */
// the Free Software Foundation; either version 2 of the License.       */
//***********************************************************************/
//																		                                  //
// Questions or comments can be left at http://forum.gizmola.com        //
//***********************************************************************/

/**
 * common.function.php is the gamestatus system function library
 * @package GameStatus chooser.class.php
 * @author David Rolston <gizmo@gizmola.com>
 * @copyright 2004, David Rolston
 */

/**
 * selectChooser creates linked selects as interface to a many-to-many relationship.
 *
 * This class should be loaded with 2 arrays of name/value pairs.  In union, these
 * items should represent the full set of items available in resolving the many to
 * many relationship.  For example, for a shirt that is available in multiple colors
 * these arrays should represent collectively the set of all possible colors.
 * These are associative arrays keyed on the description or tag which will be displayed
 * in the Select List.  The value is the key for the item which should match the key for
 * the item in the lookup list.  example:
 * <code>
 * <?php
 *  require('chooser.class.php');
 * $color = array('Red' => 1, 'Blue' => 2, 'Green' => 3, 'Black' => '4');
 * // According to database, shirt available only in Blue;
 * $r_shirt_selected = array('Blue' => 2);
 * $r_shirt_unselected = array('Red' => 1, 'Green' => 3, 'Black' => 4);
 * $shirt_selectChooser = new selectChooser($r_shirt_unselected, $r_shirt_selected);
 * //emit javascript functions
 * $shirt_selectChooser->outputjavascript();
 * //emit selectChooser
 * echo '<form>';
 * echo $shirt_selectChooser->outputChooser();
 * echo '</form>';
 * ?>
 * </code>
 * Typically these arrays would be populated by a query that outerjoined from the parent table 
 * through the many to many resolution table to the lookup table.  It is then simple to go through
 * the result and set up each array in a manner reflecting the current state of the relationship
 * as stored in the database. This provides methods to generate all HTML and javascript code required.
 */
class selectChooser {
// Properties
/**
  * @access private
  * @var The form elements for the chooser will be created based on this name. The selected list will be
	* named $listname_s[], the unselected list will be $listname_u[]. 
  */
var $listname;

/**
  * @access private
  * @var name/value associative array with unselected items from total set 
  */
var $rUnSelectL;

/**
  * @access private
  * @var name/value associative array with selected items from total set 
  */
var $rSelectL;

/**
  * @access private
  * @var total items in set (select + unselected) 
  */
var $scSize;


/**
 * php 4.x style constructor: selectChooser 
 *
 * selectChooser requires 2 associative arrays:
 * $rUnSelect (items in list that have not been selected), and $rSelect (items in list
 * that have been selected).  
 * 
 * @param reference to an associated array keyed by description, reflecting Unchosen items
 * @param reference to an associated array keyed by description, reflecting Chosen items
 * @param optional base name for the html selects generated.  Defaults to 'list'.
 */
function selectChooser(&$rUnSelect, &$rSelect, $listname='list') {
	$this->listname = $listname;
	$this->rSelectL = $rSelect;
	$this->rUnSelectL = $rUnSelect;
	$this->scSize = (count($this->rSelectL) + count($this->rUnSelectL));
}

/**
 * outputChooser 
 *
 * This is the method that outputs the select lists. 
 * 
 * @param string default = 0. For the current page DOM, this number reflects the embedded form which contains the selectChooser
 * @return string The generated selectChooser HTML is returned as a string. 
 */
function outputChooser($formnbr=0) {
	$line = "<table><tr><td>Unselected</td><td></td><td>Selected</td></tr>\n";
	$line .= "<tr><td><select name=\"".$this->listname."_u[]\" size=\"$this->scSize\" multiple>";	
	if (count($this->rUnSelectL) > 0) {
		foreach($this->rUnSelectL as $key => $value) {
			$line .= "<option value='$value'>$key</option>\n";
		}
	}
	$line .= "</select></td>";
	$line .= "<td><input type=\"button\" value=\"  >>  \" onclick=\"move(document.forms[$formnbr].elements['".$this->listname."_u[]'],document.forms[$formnbr].elements['".$this->listname."_s[]'])\"><br>";
	$line .= "<input type=\"button\" value=\"  <<  \" onclick=\"move(document.forms[$formnbr].elements['".$this->listname."_s[]'],document.forms[$formnbr].elements['".$this->listname."_u[]'])\"></td>";
	$line .= "<td><select name=\"".$this->listname."_s[]\" size=\"$this->scSize\" multiple>";
	if (count($this->rSelectL) > 0) {
		foreach($this->rSelectL as $key => $value) {
			$line .= "<option value='$value'>$key</option>\n";
		}
	}
	$line .= "</select></td></tr></table>";
	return $line;
}

/**
 * outputjavascript 
 * 
 * emits the javascript functions needed by a selectChooser.  Needs to be called prior to
 * calling selectChooser().
 * If there are multiple selectChoosers only one needs to call outputjavascript.  This would
 * be a good candidate for a static class variable.
 */
function outputjavascript() {
 $hd = <<<HEREDOC
<SCRIPT LANGUAGE="JavaScript">
<!-- Begin
sortitems = 1;  // Sort list items? (1=yes)

function move(fbox,tbox) {
	for(var i=0; i<fbox.options.length; i++) {
		if(fbox.options[i].selected && fbox.options[i].value != "") {
			var no = new Option();
			no.value = fbox.options[i].value;
			no.text = fbox.options[i].text;
			tbox.options[tbox.options.length] = no;
			fbox.options[i].value = "";
			fbox.options[i].text = "";
	   }
	}
	BumpUp(fbox);
	if (sortitems) 
		SortD(tbox);
}

function BumpUp(box)  {
	for(var i=0; i<box.options.length; i++) {
		if(box.options[i].value == "")  {
			for(var j=i; j<box.options.length-1; j++)  {
				box.options[j].value = box.options[j+1].value;
				box.options[j].text = box.options[j+1].text;
			}
			var ln = i;
			break;
   	}
	}

	if(ln < box.options.length)  {
		box.options.length -= 1;
		BumpUp(box);
   }
}

function SortD(box)  {

	var temp_opts = new Array();
	var temp = new Object();

	for(var i=0; i<box.options.length; i++)  {
		temp_opts[i] = box.options[i];
	}
	for(var x=0; x<temp_opts.length-1; x++)  {
		for(var y=(x+1); y<temp_opts.length; y++)  {
			if(temp_opts[x].text > temp_opts[y].text)  {
				temp = temp_opts[x].text;
				temp_opts[x].text = temp_opts[y].text;
				temp_opts[y].text = temp;
				temp = temp_opts[x].value;
				temp_opts[x].value = temp_opts[y].value;
				temp_opts[y].value = temp;
	    }
	  }
	}
	for(var i=0; i<box.options.length; i++)  {
		box.options[i].value = temp_opts[i].value;
		box.options[i].text = temp_opts[i].text;
	}
}

function SelectListAll(box) {
	for(var i=0; i<box.options.length; i++) {
		if(box.options[i].value != "") {
			box.options[i].selected = true;
		}
	}
	return true;
}
// End -->
</script>
HEREDOC;
 $hd .= "\n"; //clean up source a bit
 echo $hd;
 }
} // end class selectChooser
?>